// GnuRad 
// Copyright 1997, Ken Paoletti
// All rights reserved

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <assert.h>
#include "struct.h"
#include "wall.h"
#include "media.h"
#define Random  R=1.0*rand()/RAND_MAX        
#define SIGMA 5.6705E-08 //Stephan-Boltzmann constant W*m*m/K/K/K/K
#ifndef PI
#define PI 3.1415927
#endif

WALL::WALL(walldata wdat, wall_element *element, char m)
{  
    int i ;
    nel = wdat.meshx * wdat.meshy + 1 ;
    emis = new double[nel] ;
    reflect_S = new double[nel] ;
    reflect_D = new double[nel] ;
    send = new unsigned long[nel] ;
    rec = new unsigned long[nel] ;
    assert( emis != 0);
    assert( reflect_S != 0);
    assert( reflect_D != 0);
    assert( send != 0);
    assert( rec != 0);

    w_num[0] = wdat.w_num[0] ;
    w_num[1] = wdat.w_num[1] ;
    e_num[0] = wdat.e_num[0] ;
    e_num[1] = wdat.e_num[1] ;
    type = wdat.type ;
    mode = m ;
    if(type == 'U') 
      {
	q = new double[nel];
	assert(q != 0);
      }
    bun_e = wdat.bun_e ;
    bun_s = wdat.bun_s ;
    meshx = wdat.meshx ;
    meshy = wdat.meshy ;
    loc_x = wdat.loc_x ;
    loc_y = wdat.loc_y ;
    delx = loc_x/(double)meshx ;
    dely = loc_y/(double)meshy ;
    norm[0] = wdat.norm[0] ;
    norm[1] = wdat.norm[1] ;
    norm[2] = wdat.norm[2] ;
    boxX = wdat.boxl[0] ;
    boxY = wdat.boxl[1] ;
    boxZ = wdat.boxl[2] ;
    area = (loc_x/meshx)*(loc_y/meshy) ;
    max_iter = wdat.max_iter ;
    max_per = wdat.max_per ;
    iter_fac = wdat.iter_fac ;
    iter_num = 0 ;

    // The zero component is not used in these arrays
    // but will still be initailized to zero.
    emis[0] = 0 ;
    reflect_D[0] = 0 ;
    reflect_S[0] = 0 ;
    rec[0] = 0 ;

    for(i = 1; i < nel; i++)
    {
        emis[i] = element[i].emis ;
        reflect_D[i] = element[i].reflect_D ;
        reflect_S[i] = element[i].reflect_S ;
        rec[i] = 0 ;

	// View factor mode
        if(mode == 'V')
        {
            // View mode wall which is sending bundles
            if(type == 'S')
                send[i] = (unsigned long)( (double)bun_s* delx* dely) ;
            else
	      // Wall is not sending any bundles
                send[i] = 0 ;
        }
	// Temperature mode
        else if(mode == 'T')
        {
	  // The wall is a virtual wall, it does not need to send any bundles
            if(type == 'V')
                send[i] = 0 ;
	    // Temperature of the wall is unknown, initially no bundles are sent
            else if(type == 'U')
            {
                q[i] = element[i].qnet ;
                send[i] = 0 ;
            }
	    // N means NONE, the wall represents black body space at a temperature
	    // prescribed in the input.
	    // T implies the initial temperature of the wall is known
            else if( ((type == 'N')||(type == 'T'))&&(bun_e != 0) )
                send[i] = (unsigned long)(area*emis[i]*SIGMA*pow(element[i].temp,4.0)/bun_e) ;
	    // The FLUX of the wall is specified instead of the wall temperature
            else if( (type == 'F')&&(bun_e != 0))
                send[i] = (unsigned long) (area* element[i].temp/bun_e) ;
            else
            {
                fprintf(stderr,"\n\n *** Bundle energy = 0.0 check your input. ***\n\n");
                exit(3) ;
            }
        }
    }
}

// There is no copy construtor because the current design of GnuRad 
// does not require one. If there is a change it may be required and 
// will have to be added for GnuRad to work properly.

// --------- Destrutor --------------

WALL::~WALL(void)
{
  delete [] emis ;
  delete [] reflect_D ;
  delete [] reflect_S ;
  delete [] rec ;
  delete [] send ;
  if(type == 'U') delete [] q ;
}
// --------------------- Print Info -----------------

void WALL::print(FILE *fptr, char type)
{
  // Four print options are avialable, they are:
  // 1.) Table format
  // 2.) Text 'graphics' format
  // 3.) Spread Sheet format, with the columns separated by commas
  // 4.) An input file for GNU Plot

    switch(type)
    {
    case 'T':
        ptable(fptr) ;
        break ;

    case 'G':
        pgraph(fptr) ;
        break ;

    case 'S':
        pspread(fptr) ;
        break;

    case 'P':
        plot(fptr) ;
        break;
    }
}
//------------ Print the Tabulated format ------------

void WALL::ptable(FILE *fptr)
{
    int i ;
    double eout, tempS, tempR, qin, qout, qnet, qnet_sum = 0.0 ;
    extern double Qnet ;

    fprintf(fptr,"\n\n -------- BOX %i,  WALL %i PROPERTIES --------\n\n", (e_num[0] +1), (w_num[0]+1)) ;
    if(type == 'N')
    {   
        eout = bun_e* rec[1] ;
        fprintf(fptr," The wall has been removed.\n");
        if(mode == 'T')
        {
            fprintf(fptr," Total energy which passed through the plane = %9g\n", eout) ; 
        }
    }
    else
    {
        if( (type == 'T')||(type == 'U')||(type == 'F')||(type == 'R')||(type == 'P')||(type == 'S') ) 
        {
            fprintf(fptr," Wall type = %c  Model element number = %d\n", type, (e_num[0] +1 )) ;
        }
        if(type == 'V')
        {
            fprintf(fptr," Wall type = %c  Model element number = %d\n", type, (e_num[0] +1)) ;
            fprintf(fptr," Other wall number = %i, Model element number = %i\n",(w_num[1] +1),  (e_num[1]+ 1));
        }

        fprintf(fptr," Local 'x' meshed to %i, section(s)\n", meshx);
        fprintf(fptr," Local 'x' is %7g meters\n", loc_x);
        fprintf(fptr," Local 'y' meshed to %i, section(s)\n", meshy);
        fprintf(fptr," Local 'y' is %7g meters\n", loc_y);
        if(mode == 'T')  fprintf(fptr," Bundle energy = %7g\n\n",bun_e);
     
        if(type == 'U')
        {
            fprintf(fptr," Number of Iterations  %i\n", iter_num);
            fprintf(fptr," Max Iteration no. allowed %i\n", max_iter);
            fprintf(fptr," Final flux error was set to maximum of %4g percent.\n", max_per);
            fprintf(fptr," Iteration update factor = %4g\n", iter_fac);

            for(i = 1; i< nel; i++)
            {
                fprintf(fptr," Element number %i, Desired flux %5g W/m/m\n", i, q[i]);
            }
        }
        if(mode == 'T')
        {
            fprintf(fptr,"\n Energy, #Sent, and  #Received are per unit time (seconds).\n\n");
            fprintf(fptr," Elem Emis Ref_D  Ref_S   # Sent  S_temp   S_Energy  # Receive R_temp   R_Energy  Net Energy\n");
        }
        else
            fprintf(fptr,"\n Elem  Emis  Ref_D Ref_S     # Sent     # Receive\n");


        for(i = 1; i < nel; i ++)
        {
            if(emis[i] != 0.0)
            {
                tempS = pow( (bun_e* send[i]/area/emis[i]/SIGMA), 0.25 ) ;
                tempR = pow( (bun_e* rec[i]/area/emis[i]/SIGMA), 0.25 ) ;
                qin = bun_e* rec[i] ;
                qout = bun_e* (double)(send[i]);
                qnet = qout - qin ;
                qnet_sum = qnet_sum + qnet ;
                Qnet = Qnet + qnet ;

                if(mode == 'T')
                    fprintf(fptr,"%3d   %4.3f %4.3f %4.3f %9lu %#7g  %#7g   %9lu %#7g  %#7g  %#7g\n", i, emis[i], reflect_D[i], reflect_S[i], send[i], tempS, qout, rec[i], tempR, qin, qnet );
                else
                    fprintf(fptr," %3d   %4.3f %4.3f %4.3f  %9lu   %9lu\n",i, emis[i], reflect_D[i], reflect_S[i], send[i], rec[i]);
            }
            else
            {
                fprintf(fptr,"%3d   ** Reflecting surface (rho = 1.0) ** ", i );
            }
        }
        if(mode == 'T')
        {
            fprintf(fptr,"\n>>>> Net Energy for the wall = %9g\n", qnet_sum) ;
        }
    }
}
//--------------- Text-Graphics Format -------------

void WALL::pgraph(FILE *fptr)
{
    int i, j, n ;
    double tempR, qin, qout, qnet ;
 
    n = w_num[0] + 1 ;
    fprintf(fptr,"\n\n -------- BOX %i, Wall %i -------------\n\n",(e_num[0] +1 ), n) ;
    if( (type != 'V')&&(type != 'N') )
    {
        fprintf(fptr," local y\n\n ^\n |\n ");
        for(j = 1; j <= meshy; j++)
        {
            for(i = 1; i <= meshx; i++)
            {
                fprintf(fptr,"-----------");
            }
            fprintf(fptr,"\n ");
            for(i = 1; i <= meshx; i++)
            {
                n = meshx * meshy - meshx* j +i ;
                fprintf(fptr,"| El#%4i  ",n) ;
            }
            fprintf(fptr,"|\n ");
            if(type == 'U')
            {
                for(i = 1; i <= meshx; i++)
                {
                    n = meshx * meshy - meshx* j +i ;
                    if(emis[i] != 0.0)
                    {
                        tempR = pow( (bun_e* rec[n]/area/emis[i]/SIGMA), 0.25 ) ;
                        fprintf(fptr,"| %8g ",tempR);
                    }
                    else 
                    {
                        fprintf(fptr,"| *Mirror* \n") ;
                    }
                }
                fprintf(fptr,"|\n ");
            }
            for(i = 1; i <= meshx; i++)
            { 
                n = meshx * meshy - meshx* j +i ;
                qin = bun_e* rec[n] ;
                qout = bun_e* send[n] ;
                qnet = qout - qin ;
                fprintf(fptr,"| %8g ",qnet);
            }
            fprintf(fptr,"|\n ");
        }
        for(i = 0; i < meshx; i++)
        {
            fprintf(fptr,"-----------");
        }
        fprintf(fptr,"----> local x\n "); 
    }
    if(type == 'V') fprintf(fptr, " The wall is virtual\n") ;
    if(type == 'N')
    {   
        qin = bun_e* rec[1] ;
        fprintf(fptr," The wall has been removed.\n");
        fprintf(fptr," Total flux which passed through the plane = %9g\n", qin) ;
    }
}
//----------- Spread Sheet Format (csv) ------------

void WALL::pspread(FILE *fptr)
{
    double tempS, tempR, qin, qout, qnet, qnet_sum = 0.0 ;
    int i ;
    if( (type != 'V')||(type != 'N') )
    {
        fprintf(fptr,"\n*** Box %i Wall %i Information***\n", (e_num[0] + 1), (w_num[0]+ 1)) ;
        fprintf(fptr,"Bundle Energy\n %9g\n",bun_e) ;
        fprintf(fptr,"Loc x dim,%7g , Loc y dim,%7g\n", loc_x, loc_y);
        fprintf(fptr,"Meshx,%i, Meshy,%i\n", meshx, meshy);
        fprintf(fptr," Elem,Emis., Reflect_S, Reflect_D,  # Sent, S_temp, S_Energy, # Receive, R_temp, R_Energy, Net Energy\n");

        for(i = 1; i < nel; i++)
        {
            if(emis[i] != 0.0)
            {
                tempS = pow( (bun_e* send[i]/area/emis[i]/SIGMA), 0.25 ) ;
                tempR = pow( (bun_e* rec[i]/area/emis[i]/SIGMA), 0.25 ) ;
                qin = bun_e* rec[i]/area ;
                qout = bun_e* send[i]/area ;
                qnet = qout - qin ;
                qnet_sum = qnet_sum + qnet ;

                fprintf(fptr,"%3d, %4.3f, %4.3f, %4.3f, %9lu, %7g, %7g, %9lu, %7g, %7g,  %7g\n", i, emis[i], reflect_S[i], reflect_D[i], send[i], tempS, qout, rec[i], tempR, qin, qnet );
            }
            else
            {
                fprintf(fptr,"%3d  %2c ** Reflecting surface (rho = 1.0) ** ", i );

            }
        }
    }
    if(type == 'V') fprintf(fptr, " The wall is virtual\n") ;
    if(type == 'N')
    {   
        qin = bun_e* rec[1] ;
        fprintf(fptr,"\n***Wall %i Information***\n", w_num[0]) ;
        fprintf(fptr," The wall has been removed.\n");
        fprintf(fptr," Total flux which passed through the plane = ,%9g\n", qin) ;
    }
}

//--------------- GNUPLOT Format  -------------

void WALL::plot(FILE *fptr)
{
    int i, j, n ;
    double tempR, tempS, x, dx, y, dy, qin, qout, qnet ;
 
    n = w_num[0] + 1 ;
    fprintf(fptr," # BOX %i, Wall %i\n",(e_num[0] +1 ), n) ;
    if( (mode == 'T')&&(type == 'U')&&(meshx > 1)&&(meshy > 1) )
    {
        fprintf(fptr,"set parametric\nset data style lines\nset contour\n") ;
        fprintf(fptr,"set cntrparam bspline\nset xlabel 'Local x'\nset ylabel 'Local y'\n") ;
        fprintf(fptr," # Values are per unit time (seconds)\n");
        fprintf(fptr," # By default, the temp. plotted is based on the number of bunldes\n") ;
        fprintf(fptr," # received. If the plot should be based on the number sent,\n") ;
        fprintf(fptr," #  change 'using 1:2:3' to 'using 1:2:4'\n") ;
        fprintf(fptr," # If the plot should be based on energy received,\n") ;
        fprintf(fptr," #  change 'using 1:2:3' to 'using 1:2:5'\n") ;
        fprintf(fptr," # If the plot should be based on energy sent,\n") ;
        fprintf(fptr," #  change 'using 1:2:3' to 'using 1:2:6'\n") ;
        fprintf(fptr," # If the plot should be based on net energy,\n") ;
        fprintf(fptr," #  change 'using 1:2:3' to 'using 1:2:7'\n") ;
        fprintf(fptr,"splot '<file.dat>' using 1:2:3 title 'Title' \n\n") ;
        fprintf(fptr," # Cut and paste the rest of this file into a specific file\n") ;
        fprintf(fptr," # name and replace <file.dat> with that file name.\n") ;
        fprintf(fptr," # Note; keep the file name and title in single quotes\n") ;
        fprintf(fptr," #---------------- <file.dat> -------------\n\n") ;

        dx = loc_x/meshx ;
        dy = loc_y/meshy ;
        for(j = 1; j <= meshy; j++)
        {
            for(i = 1; i <= meshx; i++)
            {
                n = meshx * meshy - meshx* j +i ;
                if(emis[i] != 0.0)
                {
                    x = dx * ((double)i - 0.5) ;
                    y = dy * ((double)j - 0.5) ;
                    qin = bun_e * (double)rec[i]/area ;
                    qout = bun_e * (double)send[i]/area ;
                    qnet = qout - qin ;
                    tempS = pow( (bun_e* send[n]/area/emis[i]/SIGMA), 0.25 ) ;
                    tempR = pow( (bun_e* rec[n]/area/emis[i]/SIGMA), 0.25 ) ;
                    fprintf(fptr," %9g  %9g  %9g  %9g  %9g  %9g  %9g\n",x, y, tempR, tempS, qin, qout, qnet);
                }
                else 
                {
                    fprintf(fptr," Element %i emissivity = 0 (=> Mirror)\n") ;
                }
            }
            fprintf(fptr,"\n") ;
        }
        fprintf(fptr," ------------- End of <file.dat> -------------\n\n") ;
    }
    else
    {
        fprintf(fptr," # Wall properties are not unknown or the number\n") ;
        fprintf(fptr," # elements is not FOUR or greater.\n") ;
    }
}

// ------------- Iteratation function  -------------

// Refer to the users manual for more detail. 
// Note, all elements must be equal or less than the desired 
// percentage. 

char WALL::iterate(void)
{
    int i ;
    char ans, again = 'N';
    double percent, net, qsent, qrec ;

    if( (type == 'U')&&(iter_num < max_iter) )
    {
        iter_num++ ;
        for(i = 1; i < nel; i++)
        {
            net = q[i] * area ;
            qsent = (double)send[i] * bun_e ;
            qrec = (double)rec[i] * bun_e ;
            if( qrec != 0.0 )
                percent = fabs( fabs((qsent - net)/qrec * 100.0) - 100.0 );
            else
                percent = 100 ;
            if(percent > max_per)
            {
                assert( (bun_e != 0) );
                ans = 'Y';
                send[i] = send[i] + (unsigned long)(iter_fac*(rec[i] - send[i])) + (unsigned long)(q[i] * area/bun_e);
            }
            else
            {
                ans = 'N';
            }
            if(ans == 'Y') again = 'Y';
        }
    }
    return(again);
}

//----------- Report the wall type (.i.e real, virtual) ---------

char WALL::wtype(void) const
{
    return( type );
}

//----------- Report the other box (if wall is virtual) ----------

int WALL::other_box(void) const
{
    return( e_num[1] );
}

// ------------ Reset the # received ---------

void WALL::reset(void)
{
    int i ;

    for(i = 0; i < nel; i++)
    {
        rec[i] = 0 ;
    }
}

// ------------- Send Bundles -----------------

void WALL::sendall(void)
{
    unsigned long n ;
    int i, j, el = 1 ;
    if(type != 'V')
    {
        for(j = 0; j < meshy; j++)
        {
            for(i = 0; i < meshx; i++)
            {
                for(n = 0; n < send[el]; n++)
                {
                    bundle = intobox(w_num[0],i,j) ;
                    media->receive(bundle) ;
                }
                el++ ;
            }
        }
    }
}

// ------------- Record Bundles Sent -----------------

unsigned long WALL::sent(void) const
{
    unsigned long sent = 0 ;
    int j ;
    if(type != 'V')
    {
        for(j = 1; j < nel; j++)
        {
            sent = sent + send[j] ;
        }
    }
    return(sent) ;
}

// --------- Generate bundle info to send into the box  --------

Bundleinfo WALL::intobox(int boxwall, int i, int j)
{

    double R, theta, phi, wallx, wally ;

    // Set up the bundle starting position and direction.
    // Wall directions are changed into global(box) coordinates.
    R= 1.0*rand()/RAND_MAX;        
    wallx = delx*(R + i) ;
    R= 1.0*rand()/RAND_MAX;
    wally = dely*(R + j) ;
    R= 1.0*rand()/RAND_MAX;        
    theta= asin(sqrt(R));
    R= 1.0*rand()/RAND_MAX;                   
    phi= 2*PI*R;
    // Setting up the normal vector components and 
    // the starting position of the bundle
    switch (boxwall)
    {
    case 0:
        bundle.dir[0]= sin(theta)*cos(phi);
        bundle.dir[1]= sin(theta)*sin(phi);
        bundle.dir[2]= cos(theta);
        bundle.pos[0]= wallx;
        bundle.pos[1]= wally;
        bundle.pos[2]= 0.0;
        break; 

    case 1:
        bundle.dir[0]= sin(theta)*sin(phi);
        bundle.dir[1]= cos(theta);
        bundle.dir[2]= sin(theta)*cos(phi);
        bundle.pos[0]= wallx;
        bundle.pos[1]= 0.0;
        bundle.pos[2]= wally;
        break; 

    case 2:
        bundle.dir[0]= cos(theta);
        bundle.dir[1]= sin(theta)*cos(phi);
        bundle.dir[2]= sin(theta)*sin(phi);
        bundle.pos[0]= 0.0;
        bundle.pos[1]= wally;
        bundle.pos[2]= wallx;
        break; 

    case 3:
        bundle.dir[0]= sin(theta)*cos(phi);
        bundle.dir[1]= sin(theta)*sin(phi);
        bundle.dir[2]= -cos(theta);
        bundle.pos[0]= wallx;
        bundle.pos[1]= wally;
        bundle.pos[2]= boxZ;
        break;
      
    case 4:
        bundle.dir[0]= sin(theta)*sin(phi);
        bundle.dir[1]= -cos(theta);
        bundle.dir[2]= sin(theta)*cos(phi);
        bundle.pos[0]= wallx;
        bundle.pos[1]= boxY;
        bundle.pos[2]= wally;
        break; 

    case 5:
        bundle.dir[0]= -cos(theta);
        bundle.dir[1]= sin(theta)*cos(phi);
        bundle.dir[2]= sin(theta)*sin(phi);
        bundle.pos[0]= boxX;
        bundle.pos[1]= wally;
        bundle.pos[2]= wallx;
        break; 
    }
    bundle.optl = 0 ;
    bundle.optlext = 0.0 ;
    return(bundle); 
}
//---------- Receive a Bundle -----------

void WALL::receive(Bundleinfo bun)
{
    int i ;

    bundle = bun ;
    // IF the optical length the bundle has travelled is greater than 
    // 1x10^40 the program asserts.
    assert( (bundle.optl < 0.1E40) );
    // If the optical length to extincition is greater than 1x10^40 
    // the program asserts
    assert( (bundle.optlext < 0.1E40) ) ;
    // These three assert statements are to insure the bundle is inside
    // the box
    assert( ((boxX - bundle.pos[0]) > -0.1E-10) ) ;
    assert( ((boxY - bundle.pos[1]) > -0.1E-10) ) ;
    assert( ((boxZ - bundle.pos[2]) > -0.1E-10) ) ;

    // REAL Wall
    if(type != 'V')
    {
        bundle.optl = 0 ;
        bundle.optlext = 0.0 ;
        i = find_elem();
	// Random is a macro, it is defined differently in wall.C than in media.C
        Random ; // Generates a random number R between 0 and 1.0
	// The wall receives a bundle of energy (The photon bundle is absorbed)
        if(R <= emis[i])
        {
            rec[i]++ ;
        }
	// The wall reflects the bundle in a diffuse manner
        else if( (R > emis[i])&&(R <= (emis[i] + reflect_D[i])) )
        {
            reflect(i, 'D');
            media->receive(bundle);
        }
	// The wall reflects the bundle in a spectular manner
        else
        {
            reflect(i, 'S') ;
            media->receive(bundle) ;
        }
    }
    else //VIRTUAL Wall
    { 
        transform();
        media->receive(bundle);
    }
}

// ------------- Record Bundles Received -----------------

unsigned long WALL::received(void) const
{
    unsigned long r = 0 ;
    int j ;
    if(type == 'R')
    {
        for(j = 1; j < nel; j++)
        {
            r = r + rec[j] ;
        }
    }
    return(r) ;
}
//---------- Connect to the Media ---------

void WALL::addmedia(MEDIA *m)
{
    media = m ;
}
//-------- Find the correct element ----------

int WALL::find_elem()
{
    int i, j ;
    double x, y ;

    // The case loop is used to transform the bundle coordinates
    // which are global into local wall coordinates
    switch(w_num[0])
    {
    case 0:
        x = bundle.pos[0] ;
        y = bundle.pos[1] ;
        break;

    case 1:
        x = bundle.pos[0] ;
        y = bundle.pos[2] ;
        break;

    case 2:
        x = bundle.pos[2] ;
        y = bundle.pos[1] ;
        break;

    case 3:
        x = bundle.pos[0] ;
        y = bundle.pos[1] ;
        break;

    case 4:
        x = bundle.pos[0] ;
        y = bundle.pos[2] ;
        break;

    case 5:
        x = bundle.pos[2] ;
        y = bundle.pos[1] ;
        break;
    }
    i = 0 ;
    j = 0 ;
    // Both assert statements are sent to 10,000. This  implies
    // that the wall may be gridded up to 9,999 X 9,999 
    while( x > (delx * i) )
    {
        assert( (i < 10000) );
        i++ ;
    }
    while( y > (dely * j) )
    {
        assert( (j < 10000) );
        j++ ;
    }
    return( (i + meshx * (j-1)) );
}
//--------- Reflect the Bundle ------------

void WALL::reflect(int i, char ref)
{
    double R, theta, phi ;
    // The bundle is reflected specularly with respect the to
    // the wall
    if(ref == 'S')
    {
        switch(w_num[0])
        {
        case 0:
            bundle.dir[2] = -bundle.dir[2] ;
            break ;

        case 1:
            bundle.dir[1] = -bundle.dir[1] ;
            break ;

        case 2:
            bundle.dir[0] = -bundle.dir[0] ;
            break ;

        case 3:
            bundle.dir[2] = -bundle.dir[2] ;
            break ;

        case 4:
            bundle.dir[1] = -bundle.dir[1] ;
            break ;

        case 5:
            bundle.dir[0] = -bundle.dir[0] ;
            break ;
        }
    }
    else
    {
        // A random azmuthal?polar angle is calculated
        Random ;        
        theta= asin(sqrt(R));
	// A random azmuthal?polar angle is calculated
        Random ;                    
        phi= 2*PI*R;

	// The local random angles are transformed into global
	// normal components
        switch (w_num[0])
        {
        case 0:
            bundle.dir[0]= sin(theta)*cos(phi);
            bundle.dir[1]= sin(theta)*sin(phi);
            bundle.dir[2]= cos(theta);
            break; 

        case 1:
            bundle.dir[0]= sin(theta)*sin(phi);
            bundle.dir[1]= cos(theta);
            bundle.dir[2]= sin(theta)*cos(phi);
            break; 

        case 2:
            bundle.dir[0]= cos(theta);
            bundle.dir[1]= sin(theta)*cos(phi);
            bundle.dir[2]= sin(theta)*sin(phi);
            break; 

        case 3:
            bundle.dir[0]= sin(theta)*cos(phi);
            bundle.dir[1]= sin(theta)*sin(phi);
            bundle.dir[2]= -cos(theta);
            break;
      
        case 4:
            bundle.dir[0]= sin(theta)*sin(phi);
            bundle.dir[1]= -cos(theta);
            bundle.dir[2]= sin(theta)*cos(phi);
            break; 

        case 5:
            bundle.dir[0]= -cos(theta);
            bundle.dir[1]= sin(theta)*cos(phi);
            bundle.dir[2]= sin(theta)*sin(phi);
            break; 
        }
	// Since the bundle has had interaction, is optical length 
	// travelled and  optical length to extiction is set back
	// to zero.
        bundle.optl = 0 ;
        bundle.optlext = 0.0 ;
    }
}
//--------- Transform to other Box coord  ------------

void WALL::transform()
{
    switch(w_num[0])
    {
    case 0:
        bundle.pos[2] =  media->dimension(2) ;
        break;

    case 1:
        bundle.pos[1] = media->dimension(1) ;
        break;

    case 2:
        bundle.pos[0] = media->dimension(0) ;
        break;

    case 3:
        bundle.pos[2] = 0.0 ;
        break;

    case 4:
        bundle.pos[1] = 0.0 ;
        break;

    case 5:
        bundle.pos[0] = 0.0 ;
        break;

    }
}


