// GnuRad 
// Copyright 1997, Ken Paoletti
// All rights reserved
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>
#include <time.h>
#include <math.h>
#include "struct.h"
#include "box.h"
#include "wall.h"
#include "media.h"
#include "model.h"
#include "scan.h"
#include "parser.h"

// Construct a PARSER object which inherits from SCAN
PARSER::PARSER(char *filename)
  : SCAN(filename)
{
  ptest = 0 ;
}

// There is no copy construtor because the current design of GnuRad 
// does not require one. If there is a change it may be required and 
// will have to be added for GnuRad to work properly.

// Destroy a PARSER object.
PARSER::~PARSER(void)
{
}

// ------------------ Functions to move and know the location -----
//                    of the pointer in the input file 

// Advance the pointer in the input file by one line
void PARSER::advance(void)
{
  token = SCAN::advance();
}

// Retreive the content of the current line of the input file
char *PARSER::current_line(void)
{
  return( SCAN::current_line() );
}

// Return the current line number the pointer is on in the 
// input file
int PARSER::line_num(void)
{
  return(SCAN::line_num());
}

//---------------- Read Mode & Bundle  ----------------

void PARSER::rmode(void)
{  
  char *stopstr = "bad" ;
  double value ;
  unsigned long ivalue ;
  unsigned int seed = 1 ;
     
  advance();
  if(strcmp(token,"SEED") == 0 )
    {
      advance();
      ivalue = strtol(token,&stopstr,10);
      if(ivalue < 0) range_error();
      seed = (unsigned int)ivalue ;
      // Seeding random # by User input
      srand(seed);
      advance();
    }
  else
    srand( (unsigned)time( NULL ));

  if(strcmp(token,"MODE") == 0 )
    {
      // Seeding the random number generator by time
      advance();
      if(strcmp(token,"TEMPERATURE") == 0 )
	{
	  mode = 'T' ;
	}
      else if(strcmp(token,"VIEW") == 0 )
	{
	  mode = 'V' ;
	}
      else modeerror(); 
    }
  else kwerror(); 
  // Read the photon bundle information
  advance();
  if(strcmp(token,"BUNDLE") == 0 )
    {
      advance();
      if(mode == 'T')
	{
	  value = strtod(token,&stopstr);
          if(value <= 0.0) range_error();
	  if(strcmp(stopstr,"\0") == 0 )
	    {
	      bun_e = value ;
              bun_s = 0 ;
	    }
	  else bundleerror(); 
	}
      if(mode == 'V')
	{
	  ivalue = strtol(token,&stopstr,10);
          if(ivalue <= 0) range_error();
	  if(strcmp(stopstr,"\0") == 0 )
	    {
	      bun_s = (unsigned long)ivalue ;
	      bun_e = 0.0 ;
	    }
	  else bundleerror(); 
	}
    }
  else kwerror();
} 
//---------------- Read Model ----------------

void PARSER::rmodel(void)
{  
  char *stopstr = "bad" ;
  int ivalue ;

  advance();
  if(strcmp(token,"MODEL") == 0 )
    {
      advance();
      ivalue = (int)strtol(token,&stopstr,10);
      if(ivalue <= 0) range_error();
      if(strcmp(stopstr,"\0") == 0 )
	{
	  moddat.nbox = ivalue ;
	}
      else modelerror(); 
    }
  else kwerror();
  moddat.mode = mode ;
} 

//---------------- Read Box ----------------

void PARSER::rbox(void)
{  
  char *stopstr = "bad" ;
  double value ;
  int i, ivalue ;

  if(ptest == 0)  advance();
  if(strcmp(token,"BOX") == 0 )
    {
      advance();
      ivalue = (int)strtol(token,&stopstr,10);
      if(ivalue <= 0) range_error();
      if(strcmp(stopstr,"\0") == 0 )
	{
	  bdat.e_num = ivalue - 1 ;
	}
      else boxerror(); 
      for(i = 0; i < 3; i++)
	{
	  advance();
	  value = strtod(token,&stopstr);
          if(value <= 0.0) range_error();
	  if(strcmp(stopstr,"\0") == 0 )
	    {
	      bdat.length[i] = value ;
	    }
	  else boxerror();
	}
    }
  else kwerror();
} 
//---------------- Read Wall ----------------

void PARSER::rwall(void)
{
  char *stopstr = "bad";
  int ivalue ;

  // Defaults initialization values
  wdat.w_num[0] = 0 ;
  wdat.w_num[1] = 0 ;
  wdat.e_num[0] = 0 ;
  wdat.e_num[1] = 0 ;
  wdat.norm[0] = 0 ;
  wdat.norm[1] = 0 ;
  wdat.norm[2] = 0 ;
  wdat.meshx = 1 ;
  wdat.meshy = 1 ;
  wdat.boxl[0] = bdat.length[0] ;
  wdat.boxl[1] = bdat.length[1] ;
  wdat.boxl[2] = bdat.length[2] ;
  wdat.bun_e = bun_e ;
  wdat.bun_s = bun_s ;
        
  advance();
  if(strcmp(token,"WALL") == 0)
    {
      advance();
      ivalue = (int)strtol(token,&stopstr,10);
      if( (ivalue > 6)||(ivalue < 1) ) range_error();
      if(strcmp(stopstr,"\0") == 0)
	{
	  wdat.w_num[0] = ivalue - 1 ;
	  wdat.e_num[0] = bdat.e_num ;
	}
      else wallerror();
      setnorm();
      setlocal();
      advance();
      if(strcmp(token,"ELEMENT") == 0)
	{
	  rwall_el();
	}
      else if(strcmp(token,"TEMP") == 0)
	{
	  rtemp('T');
	  setw_el();
	}
      else if(strcmp(token,"FLUX") == 0)
	{
	  rtemp('F');
	  setw_el();
	}
      else if(strcmp(token,"UNKNOWN") == 0)
	{
	  runknown();
	  setw_el();
          riter('W');
	}
      else if(strcmp(token,"JOIN") == 0)
	{
	  rvirtual() ;
	  setw_el();
	}
      else if(strcmp(token,"PARTICIPATE") == 0)
	{
	  rview('P') ;
	  setw_el();
	}
      else if(strcmp(token,"RECEIVE") == 0)
	{
	  rview('R') ;
	  setw_el();
	}
      else if(strcmp(token,"REMOVE") == 0)
	{
          // 'N' will be used to mean NONE since 'R' already means received

	  if(mode == 'T')
	    {
	      rtemp('N');
	      setw_el();
	    }
	  else
	    {
	      rview('N') ;
	      setw_el();
	    }
	}
      else if(strcmp(token,"SEND") == 0)
	{
	  rview('S') ;
	  setw_el();
	}
      else wallerror();

    }
  else kwerror();
}
//-------------- Read Wall Elements -----------

void PARSER::rwall_el(void)
{
  
  char *stopstr = "bad" ;
  double value, sum ;
  int ivalue, i, nel ;

  advance();
  ivalue = (int)strtol(token,&stopstr,10);
  if(ivalue <= 0) range_error();
  if(strcmp(stopstr,"\0") == 0 )
    {
      wdat.meshx = ivalue ;
    }
  else wallerror(); 
  advance() ; 
  ivalue = (int)strtol(token,&stopstr,10);
  if(ivalue <= 0) range_error();
  if(strcmp(stopstr,"\0") == 0 )
    {
      wdat.meshy = ivalue ;
    }
  else wallerror();
  advance();
  if(strcmp(token,"WELEMENT") == 0)
    {
      nel = wdat.meshx * wdat.meshy + 1 ;
      welement = new wall_element[nel] ;
      assert(welement != 0);
      advance();
      if(strcmp(token,"TEMP") == 0)
        wdat.type = 'T';
      else if(strcmp(token,"FLUX") == 0)
        wdat.type = 'F';
      else if(strcmp(token,"UNKNOWN") == 0)
        wdat.type = 'U';
      else wallerror();

      for(i=1 ; i < nel; i++)
	{
	  if(wdat.type != 'U')
	    {
	      advance();
	      value = strtod(token,&stopstr);
	      if(value <= 0) range_error();
	      if(strcmp(stopstr,"\0") == 0 )
		welement[i].temp = value ;
	      else wallerror();
	    }
	  else
	    {
	      welement[i].temp = 0 ;
	    }
	  advance();
	  value = strtod(token,&stopstr);
	  if(value <= 0) range_error();
	  if(strcmp(stopstr,"\0") == 0 )
	    welement[i].emis = value ;
	  else wallerror();
	  advance();
	  value = strtod(token,&stopstr);
	  if(value < 0) range_error();
	  if(strcmp(stopstr,"\0") == 0 )
	    welement[i].reflect_D = value ;
	  else wallerror();
	  advance();
	  value = strtod(token,&stopstr);
	  if(value < 0) range_error();
	  if(strcmp(stopstr,"\0") == 0 )
	    welement[i].reflect_S = value ;
	  else wallerror();
          sum = welement[i].emis + welement[i].reflect_D + welement[i].reflect_S ;
	  if(fabs( 1.0 - sum ) > 0.001) sumerror(sum); 
  
	}   
    }
  else wallerror();
  if(wdat.type == 'U') riter('W');
}

//------------ Read Temperature -------------

void PARSER::rtemp(char wtype)
{
  char *stopstr ;
  double value, sum ;

  wdat.type = wtype ;
  if(wdat.type != 'N')
    {
      advance();
      value = strtod(token,&stopstr);
      if(value < 0.0 ) range_error();
      if(strcmp(stopstr,"\0") == 0 )
	{
	  wdat.temp = value ;
	}
      else wallerror();
      advance();
      value = strtod(token,&stopstr);
      if( (value > 1.0) || (value < 0.0) ) range_error();
      if(strcmp(stopstr,"\0") == 0 )
	{
	  wdat.emis = value ;
	}
      else wallerror();
      advance();
      value = strtod(token,&stopstr);
      if(value < 0) range_error();
      if(strcmp(stopstr,"\0") == 0 )
	wdat.reflect_D = value ;
      else wallerror();
      advance();
      value = strtod(token,&stopstr);
      if(value < 0) range_error();
      if(strcmp(stopstr,"\0") == 0 )
	wdat.reflect_S = value ;
      else wallerror();
      sum = wdat.emis + wdat.reflect_D + wdat.reflect_S ;
      if( fabs( 1.0 - sum ) > 0.001) sumerror(sum); 
    }
  else
    {
      // The wall has been removed, which is equivilent to a wall having
      // an emissivity of 1.0, and its temperature set to the temperature
      // at 
      wdat.emis = 1.0 ;
      wdat.reflect_D = 0.0 ;
      wdat.reflect_S = 0.0 ;
      advance();
      value = strtod(token,&stopstr);
      if(value < 0.0 ) range_error();
      if(strcmp(stopstr,"\0") == 0 )
	{
	  wdat.temp = value ;
	}
      else wallerror();
    }
}
//-------------- Read Unknown Wall -----------

void PARSER::runknown(void)
{
  
  char *stopstr = "bad" ;
  double value, sum ;
  int ivalue ;

  wdat.type = 'U' ;
  advance();
  ivalue = (int)strtol(token,&stopstr,10);
  if(ivalue <= 0) range_error();
  if(strcmp(stopstr,"\0") == 0 )
    {
      wdat.meshx = ivalue ;
    }
  else wallerror(); 
  advance() ; 
  ivalue = (int)strtol(token,&stopstr,10);
  if(ivalue <= 0) range_error();
  if(strcmp(stopstr,"\0") == 0 )
    {
      wdat.meshy = ivalue ;
    }
  else wallerror();
  advance();
  value = strtod(token,&stopstr);
  if(value < 0.0) range_error();
  if(strcmp(stopstr,"\0") == 0 )
    {
      wdat.emis = value ;
      wdat.temp = 0.0 ;
    }
  else wallerror();
  advance();
  value = strtod(token,&stopstr);
  if(value < 0) range_error();
  if(strcmp(stopstr,"\0") == 0 )
    wdat.reflect_D = value ;
  else wallerror();
  advance();
  value = strtod(token,&stopstr);
  if(value < 0) range_error();
  if(strcmp(stopstr,"\0") == 0 )
    wdat.reflect_S = value ;
  else wallerror();
  sum = wdat.emis + wdat.reflect_D + wdat.reflect_S ;
  if( fabs( 1.0 - sum ) > 0.001) sumerror(sum); 
}

//------- Read Iteration information -------

void PARSER::riter(char type)
{ 
  char *stopstr = "bad" ;
  double value ;
  int ivalue, nel, i ;

  advance(); 
  if(strcmp(token,"ITERATE") == 0)
    {
      advance();
      ivalue = (int)strtol(token,&stopstr,10);
      if(ivalue < 0) range_error();
      if(strcmp(stopstr,"\0") == 0 )
	{
	  if(type == 'W')
	    wdat.max_iter = ivalue;
	  else
	    mdat.max_iter = ivalue;
	}
      else itererror();
      advance();
      value = strtod(token,&stopstr);
      if(value < 0.0) range_error();
      if(strcmp(stopstr,"\0") == 0 )
	{
	  if(type == 'W')
	    wdat.max_per = value;
	  else
	    mdat.max_per = value;
	}
      else itererror();
      advance();
      value = strtod(token,&stopstr);
      if(value < 0.0) range_error();
      if(strcmp(stopstr,"\0") == 0 )
	{
	  if(type == 'W')
	    wdat.iter_fac = value;
	  else
	    mdat.iter_fac = value;
	}
      else itererror();
      if(type == 'W')
	nel = wdat.meshx * wdat.meshy +1 ;
      else 
	nel = mdat.meshx * mdat.meshy * mdat.meshz +1 ;
      for(i = 1; i < nel; i++)
	{
	  advance();
	  value = strtod(token,&stopstr);
	  if(strcmp(stopstr,"\0") == 0 )
	    {
	      if(type == 'W')  
		welement[i].qnet = value ;
	      else
		melement[i].qnet = value ;
	    }
	  else itererror();
	}

    }
  else kwerror();
}

//---------- Read Virtual Wall -------------

void PARSER::rvirtual(void)
{

  char *stopstr = "bad" ;
  int ivalue ;

  wdat.type = 'V' ;
  wdat.temp = 0.0 ;
  wdat.emis = 1.0 ;
  advance();
  ivalue = (int)strtol(token,&stopstr,10);
  if(ivalue <= 0) range_error();
  if(strcmp(stopstr,"\0") == 0 )
    {
      wdat.e_num[1] = ivalue - 1 ;
    }
  else wallerror(); 
  advance() ; 
  ivalue = (int)strtol(token,&stopstr,10);
  if(ivalue <= 0) range_error();
  if(strcmp(stopstr,"\0") == 0 )
    {
      wdat.w_num[1] = ivalue - 1 ;
    }
  else wallerror();
}
//---------- Read View Factor -------------

void PARSER::rview(char type)
{
  char *stopstr = "bad" ;
  double value, sum ;

  wdat.type = type ;
  wdat.temp = 0.0 ; 

  if( (type == 'P')||(type == 'R')||(type == 'S') )
    {
      advance();
      value = strtod(token,&stopstr);
      if( (value < 0.0)||(value > 1.0) ) range_error();
      if(strcmp(stopstr,"\0") == 0 )
	{
	  wdat.emis = value ;
	}
      else wallerror();
      advance();
      value = strtod(token,&stopstr);
      if(value < 0) range_error();
      if(strcmp(stopstr,"\0") == 0 )
	wdat.reflect_D = value ;
      else wallerror();
      advance();
      value = strtod(token,&stopstr);
      if(value < 0) range_error();
      if(strcmp(stopstr,"\0") == 0 )
	wdat.reflect_S = value ;
      else wallerror();
      sum = wdat.emis + wdat.reflect_D + wdat.reflect_S ;
      if( fabs( 1.0 - sum ) > 0.001) sumerror(sum); 
      if(type == 'S')
	{
	  wdat.bun_s = bun_s ;
	}
    }
  else if(type == 'N')
    {
      wdat.emis = 1.0;
    }
  else wallerror();
}
//--------- Set wall unit norm --------

void PARSER::setnorm(void)
{
  if(wdat.w_num[0] == 1) wdat.norm[2] =  1.0; // Z, k =  1.0
  if(wdat.w_num[0] == 2) wdat.norm[1] = -1.0; // Y, j = -1.0
  if(wdat.w_num[0] == 3) wdat.norm[0] =  1.0; // X, i =  1.0
  if(wdat.w_num[0] == 4) wdat.norm[2] = -1.0; // Z, k = -1.0
  if(wdat.w_num[0] == 5) wdat.norm[1] =  1.0; // Y, j =  1.0
  if(wdat.w_num[0] == 6) wdat.norm[0] = -1.0; // X, i = -1.0
}
//--------- Set wall local coord --------

void PARSER::setlocal(void)
{
  if((wdat.w_num[0] == 0) || (wdat.w_num[0] == 3) )
    {
      wdat.loc_x = bdat.length[0] ;
      wdat.loc_y = bdat.length[1] ;
    }
  if((wdat.w_num[0] == 1) || (wdat.w_num[0] == 4) )
    {
      wdat.loc_x = bdat.length[0] ;
      wdat.loc_y = bdat.length[2] ;
    }
  if((wdat.w_num[0] == 2) || (wdat.w_num[0] == 5) )
    {
      wdat.loc_x = bdat.length[2] ;
      wdat.loc_y = bdat.length[1] ;
    }
}
//------- Set up Wall elements --------
void PARSER::setw_el(void)
{
  int i, nel ;
  nel = wdat.meshx * wdat.meshy + 1 ;
  welement = new wall_element[nel] ;
  assert(welement != 0);

  for(i=1 ; i < nel; i++)
    {
      welement[i].temp = wdat.temp ;
      welement[i].emis = wdat.emis ;
      welement[i].reflect_D = wdat.reflect_D ;
      welement[i].reflect_S = wdat.reflect_S ;
    }
}
//--------- Read Media ---------------

void PARSER::rmedia(void)
{  
  char *stopstr = "bad" ;
  int ivalue ;

  mdat.mode = mode ;
  advance();
  if(strcmp(token,"MEDIA") == 0 )
    {
      advance();
      ivalue = (int)strtol(token,&stopstr,10);
      if(ivalue <= 0) range_error();
      if(strcmp(stopstr,"\0") == 0 )
	{
	  mdat.e_num  = ivalue ;
	}
      else mederror(); 
      advance();
      if(strcmp(token,"CLEAR") == 0 )
	{
	  rclear();
          set_med_el() ;
	}
      else if(strcmp(token,"ELEMENT") == 0 )
	{
	  rmed_el();
	}
      else if(strcmp(token,"TEMP") == 0 )
	{
	  rmed('T');
          set_med_el() ;
	}
      else if(strcmp(token,"FLUX") == 0 )
	{
	  rmed('F');
          set_med_el() ;
	}
      else if(strcmp(token,"PARTICIPATE") == 0 )
	{
	  rmed('P');
          set_med_el() ;
	}
      else if(strcmp(token,"UNKNOWN") == 0 )
	{
	  rmedunk();
          set_med_el() ;
	  riter('M');
	}
      else mederror();
      mdat.X = bdat.length[0] ;
      mdat.Y = bdat.length[1] ;
      mdat.Z = bdat.length[2] ;
      mdat.bun_e = bun_e ;
    }
  else kwerror();
}
//----------- Media Clear -----------

void PARSER::rclear(void)
{
  mdat.temp = 0.0 ;
  mdat.Ke = 0.0 ;
  mdat.ssal = 1.0 ;
  mdat.meshx = 1 ;
  mdat.meshy = 1 ;
  mdat.meshz = 1 ;
  mdat.type = 'C' ;
  mdat.scat = 'C' ;
}
//-------------- Read Media Elements -----------

void PARSER::rmed_el(void)
{
  
  char *stopstr = "bad" ;
  double value ;
  int ivalue, i, nel ;

  advance();
  ivalue = (int)strtol(token,&stopstr,10);
  if(ivalue <= 0) range_error();
  if(strcmp(stopstr,"\0") == 0 )
    {
      mdat.meshx = ivalue ;
    }
  else mederror(); 
  advance() ; 
  ivalue = (int)strtol(token,&stopstr,10);
  if(ivalue <= 0) range_error();
  if(strcmp(stopstr,"\0") == 0 )
    {
      mdat.meshy = ivalue ;
    }
  else mederror();
  advance();
  ivalue = (int)strtol(token,&stopstr,10);
  if(ivalue <= 0) range_error();
  if(strcmp(stopstr,"\0") == 0 )
    {
      mdat.meshz = ivalue ;
    }
  else mederror();
  advance();
  value = strtod(token,&stopstr);
  if(value <= 0.0 ) range_error();
  if(strcmp(stopstr,"\0") == 0 )
    {
      mdat.Ke = value ;
    }
  else mederror();
  advance();
  value = strtod(token,&stopstr);
  if(strcmp(stopstr,"\0") == 0 )
    {
      mdat.ssal = value ;
    }
  else mederror();
  advance();
  if(strcmp(token,"ISOTROPIC") == 0 )
    mdat.scat = 'I' ;
  else if(strcmp(token,"S2") == 0 )
    mdat.scat = '2' ;
  else if(strcmp(token,"S3") == 0 )
    mdat.scat = '3' ;

  else mederror() ;
  advance();
  if(strcmp(token,"MELEMENT") == 0)
    {
      nel = mdat.meshx * mdat.meshy* mdat.meshz + 1 ;
      melement = new med_element[nel] ;
      assert(melement != 0);
      advance();
      if(strcmp(token,"TEMP") == 0)
        mdat.type = 'T';
      else if(strcmp(token,"FLUX") == 0)
        mdat.type = 'F';
      else mederror();

      for(i=1 ; i < nel; i++)
	{
	  advance();
	  value = strtod(token,&stopstr);
	  if(value <= 0) range_error();
	  if(strcmp(stopstr,"\0") == 0 )
	    melement[i].temp = value ;
	  else mederror();
	}   
    }
  else mederror();
}



//----------- Read  Media With Known Temp -----

void PARSER::rmed(char mtype)
{
  char *stopstr = "bad" ;
  double value ;
 
  mdat.type = mtype ;
  if(mode != 'V')
    {
      advance();
      value = strtod(token,&stopstr);
      if(value < 0.0) range_error();
      if(strcmp(stopstr,"\0") == 0 )
	{
	  mdat.temp = value ;
	}
      else mederror();
    }
  advance();
  value = strtod(token,&stopstr);
  if(value < 0.0) range_error();
  if(strcmp(stopstr,"\0") == 0 )
    {
      mdat.Ke = value ;
    }
  else mederror();
  advance();
  value = strtod(token,&stopstr);
  if( (value < 0.0) || (value >1.0) )range_error();
  if(strcmp(stopstr,"\0") == 0 )
    {
      mdat.ssal = value ;
    }
  else mederror() ;
  advance();
  if(strcmp(token,"ISOTROPIC") == 0 )
    mdat.scat = 'I' ;
  else if(strcmp(token,"S2") == 0 )
    mdat.scat = '2' ;
  else if(strcmp(token,"S3") == 0 )
    mdat.scat = '3' ;
  else mederror() ;

  mdat.meshx = 1 ;
  mdat.meshy = 1 ;
  mdat.meshz = 1 ;
}
//----------- Read Unknown Media -----

void PARSER::rmedunk(void)
{
  char *stopstr = "bad" ;
  double value ;
  int ivalue ;

  mdat.temp = 0.0 ;
  advance();
  ivalue = (int)strtol(token,&stopstr,10);
  if(ivalue <= 0) range_error();
  if(strcmp(stopstr,"\0") == 0 )
    {
      mdat.meshx = ivalue ;
    }
  else mederror();
  advance();
  ivalue = (int)strtol(token,&stopstr,10);
  if(ivalue <= 0) range_error();
  if(strcmp(stopstr,"\0") == 0 )
    {
      mdat.meshy = ivalue ;
    }
  else mederror();
  advance();
  ivalue = (int)strtol(token,&stopstr,10);
  if(ivalue <= 0) range_error();
  if(strcmp(stopstr,"\0") == 0 )
    {
      mdat.meshz = ivalue ;
    }
  else mederror();
  advance();
  value = strtod(token,&stopstr);
  if (value < 0.0) range_error();
  if(strcmp(stopstr,"\0") == 0 )
    {
      mdat.Ke = value ;
    }
  else mederror();
  advance();
  value = strtod(token,&stopstr);
  if( (value < 0.0) || (value >1.0) )range_error();
  if(strcmp(stopstr,"\0") == 0 )
    {
      mdat.ssal = value ;
    }
  else mederror();
  mdat.type = 'U' ;
  advance();
  if(strcmp(token,"ISOTROPIC") == 0 )
    mdat.scat = 'I' ;
  else if(strcmp(token,"S2") == 0 )
    mdat.scat = '2' ;
  else if(strcmp(token,"S3") == 0 )
    mdat.scat = '3' ;
  else mederror() ;
}
//------- Set up media elements --------

void PARSER::set_med_el(void)
{
  int i, nel ;
  nel = mdat.meshx * mdat.meshy * mdat.meshz + 1 ;
  melement = new med_element[nel] ;
  assert(melement != 0);
  for(i=1 ; i < nel; i++)
    {
      melement[i].temp = mdat.temp ;
    }
}
//--------  Read Particle ----------

void PARSER::rpart(void)
{
  char *stopstr = "bad" ;
  double value ;
  int i ;

  for( i = 0; i < 6; i++)
    {
      mdat.p[i] = 0.0 ;
    }

  advance();
  if(strcmp(token,"PARTICLE")== 0)
    { 
      for( i = 0; i < 6; i++)
	{
	  advance();
	  value = strtod(token,&stopstr);
	  if(strcmp(stopstr,"\0") == 0 )
	    {
	      mdat.p[i] = value ;
	    }
	  else parterror();
	}
      ptest = 0 ;
    }
  else ptest = 1 ;
}

//-----------------  Error Routines ----------------

void PARSER::kwerror(void)
{
  char *temp ;
  temp = current_line();
  lnum = line_num();
  fprintf(stderr,"\n ***** Token error *****\n\n");
  fprintf(stderr," Error on line %i ->%s",lnum,temp);
  fprintf(stderr," The token is either not a key word(Mode,Model,Box, Wall, ..etc)\n");
  fprintf(stderr," or it is out of sequence.\n");
  fprintf(stderr," Token was, %s\n",token);
  exit(1);
}
void PARSER::modeerror(void)
{
  char *temp ;
  temp = current_line();

  lnum = line_num();
  fprintf(stderr,"\n ***** Error in reading in the mode data *****\n\n");
  fprintf(stderr," Error on line %i ->%s",lnum,temp);
  fprintf(stderr," Last value read (will appear in upper case), %s\n", token);
  exit(1);
}
void PARSER::modelerror(void)
{
  char *temp ;
  temp = current_line();

  lnum = line_num();
  fprintf(stderr,"\n ***** Error in reading in the model data *****\n\n");
  fprintf(stderr," Error on line %i ->%s",lnum,temp);
  fprintf(stderr," Last value read (will appear in upper case), %s\n", token);
  exit(1);
}
void PARSER::bundleerror(void)
{
  char *temp ;
  temp = current_line();

  lnum = line_num();
  fprintf(stderr,"\n ***** Error in reading the bundle data *****\n\n");
  fprintf(stderr," Error on line %i ->%s",lnum,temp);
  fprintf(stderr," Last value read (will appear in upper case), %s\n", token);
  exit(1);
}
void PARSER::boxerror(void)
{
  char *temp ;
  
  temp = current_line();
  lnum = line_num();
  fprintf(stderr,"\n ***** Error in reading the box data *****\n\n");
  fprintf(stderr," Error on line %i ->%s",lnum,temp);
  fprintf(stderr," Last value read (will appear in upper case), %s\n", token);
  exit(1);
}

void PARSER::wallerror(void)
{
  char *temp ;
  temp = current_line();
  int  temp2 = wdat.w_num[0] +1 ;

  lnum = line_num();
  fprintf(stderr,"\n ***** Error in reading the wall data *****\n\n");
  fprintf(stderr," Error on line %i ->%s",lnum,temp);
  fprintf(stderr," Last value read (will appear in upper case), %s\n", token);
  exit(1);
}
void PARSER::mederror(void)
{
  char *temp ;
  temp = current_line();

  lnum = line_num();
  fprintf(stderr,"\n ***** Error in reading the media data *****\n\n");
  fprintf(stderr," Error on line %i ->%s",lnum,temp);
  fprintf(stderr," Last value read (will appear in upper case), %s\n", token);
  exit(1);
}
void PARSER::parterror(void)
{
  char *temp ;
  temp = current_line();

  lnum = line_num();
  fprintf(stderr,"\n ***** Error in reading the particle data *****\n\n");
  fprintf(stderr," Error on line %i ->%s",lnum,temp);
  fprintf(stderr," Last value read (will appear in upper case), %s\n", token);
  exit(1);
}
void PARSER::itererror(void)
{
  char *temp ;
  temp = current_line();
  int  temp2 = wdat.w_num[0] +1 ;

  lnum = line_num();
  fprintf(stderr,"\n ***** Error in reading the iteration data *****\n\n");
  fprintf(stderr," Error on line %i ->%s",lnum,temp);
  fprintf(stderr," Last value read (will appear in upper case), %s\n", token);
  exit(1);
}

void PARSER::range_error(void)
{
  char *temp ;
  temp = current_line();
  int temp2 = wdat.w_num[0] +1 ;

  lnum = line_num();

  fprintf(stderr,"\n ***** Range Error *****\n\n");
  fprintf(stderr," Error on line %i ->%s",lnum,temp);
  fprintf(stderr," The value specified is outside of its limit\n");
  exit(1);
}

void PARSER::sumerror(double value)
{
  
  lnum = line_num();

  fprintf(stderr," The sum of the emissivity, diffuse reflectivity,\n");
  fprintf(stderr," and the specular reflectivity does not sum to 1.0.\n");
  fprintf(stderr," Please correct.\n");
  fprintf(stderr," Actual sum = %f\n", value);
  fprintf(stderr," Error around line %i\n",lnum); 
  exit(1);
}

//-------------- Build the model -----------------

MODEL *PARSER::Build_model(void)
{
  int i, j ;
  rmode();
  rmodel();
  MODEL *model = new MODEL(moddat);
  assert(model != 0);

  for(j=0 ; j < moddat.nbox ; j++)
    {
      rbox();
      BOX *box =  new BOX(bdat) ;
      assert(box != 0);
      for(i= 0 ; i< 6 ; i++)
        {
	  rwall();
	  WALL *wall = new WALL(wdat,welement, mode);
	  assert(wall != 0);
	  box->add_w(wall, i);
	  wall = 0 ;
	  delete [] welement ;
	}
      rmedia();
      rpart();
      MEDIA *media = new MEDIA(mdat,melement);
      assert(media != 0);
      box->add_m(media);
      media = 0 ; 
      delete [] melement ;
      model->addbox(box, j);
      box = 0 ;
    }
  return(model);
}
