// GnuRad 
// Copyright 1997, Ken Paoletti
// All rights reserved
#include <stdio.h>
#include "struct.h"
#include "wall.h"
#include "media.h"
#include "box.h"
#include "model.h"
#include "assert.h"

//--------- Construct a Model ---------

MODEL::MODEL(modeldata moddat)
{
  m = moddat.mode ;
  nbox = moddat.nbox ;
  
  box = new BOX *[nbox] ;
  assert(box != 0);
}

// There is no copy construtor because the current design of GnuRad 
// does not require one. If there is a change it may be required and 
// will have to be added for GnuRad to work properly.

//-------- Destructor -------------

MODEL::~MODEL(void)
{
  delete [] box ;
}

//---------- Add a Box ------------

void MODEL::addbox(BOX *b, int i) 
{
  box[i] = b ;
  box[i]->join();
}

//--------- Report the mode ----------

char MODEL::mode(void)
{
  return(m) ;
}

//-------- Join the boxes together ------

void MODEL::connect()
{
  char check ;
  int i, wnum, jbox ;
  MEDIA *mptr ;

  if(nbox >= 2)
    {
      for(i = 0; i < nbox; i++)
		{
		  for(wnum = 0; wnum < 6; wnum++)
			{
			  check = box[i]->wall_type(wnum);
			  if(check == 'V') // This is a virtual wall, it is where two
				// enclosures meet. If this is the case both enclosures must
				// have access to the others media. This is done via pointers 
				{
				jbox = box[i]->joining_box(wnum);
				mptr = box[jbox]->media_ptr() ;
				box[i]->join_boxes(wnum,mptr);
				}
			}
		}
    }
}

//--------- Run the model ---------

void MODEL::run(void)
{
  int i, iter = 0 ;
  char ans, again = 'N';

  do
    {
      if(again == 'Y')
	{
	  for(i = 0; i < nbox; i++)
	    {
	      box[i]->reset();
	    }
          iter++ ;
	  fprintf(stdout," Iteration = %i\n",iter);
	  again = 'N' ;
	}

      for(i = 0; i < nbox; i++)
	{
	  fprintf(stdout," Box %i sending all bundles\n",(i+1) );
	  box[i]->sendall();
	}

      for(i = 0; i < nbox; i++)
	{
	  ans = box[i]->iterate();
	  if(ans == 'Y') again = 'Y';
	}
    }while(again == 'Y');
}

//----- Bundles Sent in View Factor Mode ----

void MODEL::pview(FILE *fptr, char type)
{
  unsigned long sent = 0 ;
  unsigned long r = 0 ;
  int i ;
  double vf ;

  for(i = 0; i < nbox; i++)
    {
      sent = sent + box[i]->sent() ;
    }

  for(i = 0; i < nbox; i++)
    {
      r = r + box[i]->received() ;
    }

  vf = (double)r/sent ;
  fprintf(fptr,"\nView or Radiation Exchange Factor = %9g\n",vf);
}

//-------- Print -----------

void MODEL::print(FILE *fptr, char type)
{
  int i ;

  for(i = 0; i < nbox; i++)
    {
      box[i]->print(fptr, type);
    }
}
