// GnuRad 
// Copyright 1997, Ken Paoletti
// All rights reserved
#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <assert.h>
#include "struct.h"
#include "wall.h"
#include "media.h"
#define SIGMA 5.6705E-08 //Stephan-Boltzmann constant W*m*m/K/K/K/K
#define Random  (1.0*rand()/RAND_MAX)
#define MAX_TIMES 30000   
#ifndef PI     
#define PI 3.1415927
#endif  
#ifndef false
#define false 0
#endif

MEDIA::MEDIA(mediadata mdat, med_element *element)
{

  int i ;
  for(i = 0; i < 6; i++)
    {
      wall[i] = 0;
    }
  nel = mdat.meshx * mdat.meshy * mdat.meshz + 1;
  send = new unsigned long[nel] ;
  rec = new unsigned long[nel] ;
  assert(send != 0);
  assert(rec != 0);

  e_num = mdat.e_num ;
  mode = mdat.mode ;
  type = mdat.type ;
  if(type == 'U') 
    {
      qm = new double[nel];
      assert(qm != 0);
    }
  scat = mdat.scat ;
  bun_e = mdat.bun_e ;
  meshx = mdat.meshx ;
  meshy = mdat.meshy ;
  meshz = mdat.meshz ;
  dim[0] = mdat.X ;
  dim[1] = mdat.Y ;
  dim[2] = mdat.Z ;
  delx = dim[0]/meshx ;
  dely = dim[1]/meshy ;
  delz = dim[2]/meshz ;
  volume = (dim[0]/meshx)*(dim[1]/meshy)*(dim[2]/meshz) ;
  Ke = mdat.Ke ;
  ssal = mdat.ssal ;
  for(i = 0; i < 6; i++)
    {
      p[i] = mdat.p[i] ;
    }

  max_iter = mdat.max_iter ;
  max_per = mdat.max_per ;
  iter_fac = mdat.iter_fac ;
  iter_num = 0 ;

  for(i = 1; i < nel; i++)
    {
      rec[i] = 0 ;
      if(mode == 'V')
	{
	  send[i] = 0 ;
	}
      else if(mode == 'T')
	{
	  if(type == 'C')
	    send[i] = 0 ;
	  else if(type == 'U')
	    {
	      qm[i] = element[i].qnet;
	      send[i] = 0 ;
	    }
	  else if( (type == 'T')&&(bun_e != 0.0) )
	    send[i] = (unsigned long)(4 * volume * (1-ssal) * Ke * SIGMA 
				      * pow(element[i].temp,4.0)/bun_e) ;
	  else if( (type == 'F')&&(bun_e != 0.0) )
	    send[i] = (unsigned long)(volume * element[i].temp/bun_e) ;
	  else
	    fprintf(stderr,"\n\n *** Bundle energy = 0.0 check your input. ***\n\n");
	  
	}

    }
}

// There is no copy construtor because the current design of GnuRad 
// does not require one. If there is a change it may be required and 
// will have to be added for GnuRad to work properly.

// ---------- Destructor ---------------

MEDIA::~MEDIA(void)
{
  delete [] rec ;
  delete [] send ;
  if(type == 'U') delete [] qm ;
}

// --------------------- Print Info -----------------

void MEDIA::print(FILE *fptr, const char type) 
{

  switch(type)
    {
    case 'T':
      ptable(fptr) ;
      break ;

    case 'G':
      pgraph(fptr) ;
      break ;

    case 'S':
      pspread(fptr) ;
      break;

    case 'P':
      //nothing right now ;
      break;
    }
}
//------------ Print the Table format ------------

void MEDIA::ptable(FILE *fptr)
{
  int i ;
  double tempS, tempR, qin, qout, qnet, qnet_sum = 0.0 ;
  extern double Qnet ;

  fprintf(fptr,"\n\n -------- BOX %i MEDIA PROPERTIES ---------\n\n", e_num) ;
  if(type != 'C')
    {
      fprintf(fptr," Media Type = %c  Scatter = %c Model element number = %d \n", type, scat, e_num) ;

      fprintf(fptr," 'X' meshed to %i, section(s)\n", meshx);
      fprintf(fptr," 'X' is %7g meters\n", dim[0]);
      fprintf(fptr," 'Y' meshed to %i, section(s)\n", meshy);
      fprintf(fptr," 'Y' is %7g meters\n", dim[1]);
      fprintf(fptr," 'Z' meshed to %i, section(s)\n", meshz);
      fprintf(fptr," 'Z' is %7g meters\n", dim[2]);
      if(mode == 'T')  fprintf(fptr," Bundle energy = %7g\n",bun_e);
      fprintf(fptr," Extinction Coefficient (Ke) = %7g \n", Ke);
      fprintf(fptr," Single scattering albedo (omega) = %4.3f\n\n", ssal);

      if(type == 'U')
	{
	  fprintf(fptr," Number of Iterations  %i\n", iter_num);
	  fprintf(fptr," Max Iteration no. allowed %i\n", max_iter);
	  fprintf(fptr," Final flux error was set to maximum of %4g percent.\n", max_per);
	  fprintf(fptr," Iteration update factor = %5g\n", iter_fac);

	  for(i = 1; i< nel; i++)
	    {
	      fprintf(fptr," Element number %i, Desired flux %5g W/m/m/m\n", i, qm[i]);
	    }
	}
 

      if((Ke != 0.0)&&(ssal != 1.0) )
	{
	  if(mode == 'T')
	    {
	      fprintf(fptr,"\n Energy, #Sent, and #Received are per unit time (seconds).\n");

	      fprintf(fptr," Elem   # Sent    (temp) S_Energy   # Received (temp)  R_Energy  Net Energy\n");
	    }
	  else
	    fprintf(fptr," Elem    # Sent    # Receive\n");

	  for(i = 1; i < nel; i ++)
	    { 
	      qin = bun_e* rec[i] ;
	      qout = bun_e* send[i] ;
	      qnet = qout - qin ;
	      qnet_sum = qnet_sum + qnet ;
	      Qnet = Qnet + qnet ;
	      tempS = pow( (bun_e* send[i]/4.0/volume/(1.0- ssal)/Ke/SIGMA), 0.25);
	      tempR = pow( (bun_e* rec[i]/4.0/volume/(1.0- ssal)/Ke/SIGMA), 0.25);

	      if(mode == 'T')
		fprintf(fptr,"%3d  %9lu  %#7g  %#7g  %9lu   %#7g  %#7g  %#7g\n", i, send[i], tempS, qout, rec[i], tempR, qin, qnet );
	      else
		fprintf(fptr," %3d  %9lu   %9lu\n",i, send[i], rec[i]);

	    }
	  if( mode != 'V')
	    {
	      fprintf(fptr,"\n>>>> Net Energy for the Media = %9g\n", qnet_sum) ;
	    }
	}
      else
	{
	  if(ssal == 1.0)
	    {
	      fprintf(fptr, " The Media is pure scattering (Single scattering albedo = 1.0)\n");
	    }
	  if(Ke == 0.0)
	    {
	      fprintf(fptr, " The Media extinction coefficient (Ke) = 0\n");
	    }
	}
    }
  else 
    {
      fprintf(fptr," The Media is Clear !\n");
    }
}
//--------------- Text Media -------------

void MEDIA::pgraph(FILE *fptr)
{
  int i, j, k, n ;
  double tempR, qin, qout, qnet ;
  if(type != 'C')
    {
      if( (Ke != 0.0)&&(ssal != 1.0) )
	{
	  fprintf(fptr,"\n\n -------- BOX %i MEDIA PROPERTIES -------\n",e_num);
	  for(k = 0; k < meshz; k++)
	    {
	      //fprintf(fptr,"\n Z Mesh number = %i",(meshz -k) );
	      if((k == 0)&& (nel > 2) )
		fprintf(fptr,"\n (Top)\n") ;
	      else if((k == (meshz - 1))&& (nel > 2))
		fprintf(fptr,"\n (Bottom)\n") ;
	      else
		fprintf(fptr,"\n\n") ;
	      fprintf(fptr," Y\n\n ^\n |\n ");
	      for(j = 1; j <= meshy; j++)
		{
		  for(i = 1; i <= meshx; i++)
		    {
		      fprintf(fptr,"-----------");
		    }
		  fprintf(fptr,"\n ");
		  for(i = 1; i <= meshx; i++)
		    {
		      n = meshx* meshy*meshz -k*(meshx * meshy) - meshx* j +i ;
		      fprintf(fptr,"| El#%4i  ",n);
		    }
		  fprintf(fptr,"|\n ");
		  if(type == 'U')
		    {
		      for(i = 1; i <= meshx; i++)
			{
			  n = meshx* meshy*meshz -k*(meshx * meshy) - meshx* j +i ;
			  tempR = pow( (bun_e* rec[n]/4.0/volume/(1.0- ssal)/Ke/SIGMA), 0.25);
			  fprintf(fptr,"|  %7g ",tempR);
			}
		      fprintf(fptr,"|\n ");
		    }
		  for(i = 1; i <= meshx; i++)
		    {
		      n = meshx* meshy*meshz -k*(meshx * meshy) - meshx* j +i ;
		      qout = bun_e* send[n] ;
		      qin = bun_e* rec[n] ;
		      qnet = qin - qout ;
		      fprintf(fptr,"|  %7g ",qnet);
		    }
		  fprintf(fptr,"|\n ");
		}
	      for(i = 0; i < meshx; i++)
		{
		  fprintf(fptr,"------------");
		}
	      fprintf(fptr,"----> X\n");
	    }
	}
      else  
	{
	  if(ssal == 1.0)
	    {
	      fprintf(fptr, "\n\n The Media is pure scattering (Single scattering albedo = 1.0)\n");
	    }
	  if(Ke == 0.0)
	    {
	      fprintf(fptr, "\n\n The Media extinction coefficient (Ke) = 0\n");
	    }
	}

    }
  else
    {
      fprintf(fptr,"\n\n The Media is Clear.\n");
    }
}

//----------- Spread Sheet Output ------------

void MEDIA::pspread(FILE *fptr)
{
  int i ;
  if(type != 'C')
    {
      if( (Ke != 0.0)&&(ssal != 1.0) )
	{
	  fprintf(fptr,"***Box %i Media Information***\n",e_num) ;
	  fprintf(fptr,"Bundle Energy\n %9g\n",bun_e) ;
	  fprintf(fptr,"Xdim,%7g , Ydim,%7g, Zdim,%7g\n",dim[0], dim[1], dim[2]);
	  fprintf(fptr,"Meshx,%i, Meshy,%i, Meshz,%i\n", meshx, meshy, meshz);
	  fprintf(fptr,"Element, #Sent, #Receive\n") ;

	  for(i = 1; i < nel; i++)
	    {
	      fprintf(fptr," %i, %i, %i\n", i, send[i], rec[i]) ;
	    }
	}  
      else  
	{
	  if(ssal == 1.0)
	    {
	      fprintf(fptr, " The Media is pure scattering (Single scattering albedo = 1.0)\n");
	    }
	  if(Ke == 0.0)
	    {
	      fprintf(fptr, " The Media extinction coefficient (Ke) = 0\n");
	    }
	}
    }
  else
    {
      fprintf(fptr,"\n\n The Media is Clear !\n");
    }

}
// ------------- Send Bundles -----------------

void MEDIA::sendall(void) 
{
  unsigned long n ;
  int i, j, k, el = 1 ;

  for(k = 0; k < meshz; k++)
    {
      for(j = 0; j < meshy; j++)
	{
	  for(i = 0; i < meshx; i++)
	    {
	      for(n = 0; n < send[el]; n++)
		{
		  bundle.pos[0] = delx*(Random + i) ;
		  bundle.pos[1] = dely*(Random + j) ;
		  bundle.pos[2] = delz*(Random + k) ;
		  bundle.optl = 0.0 ;
		  bundle.optlext = 0.0 ;
		  if(scat == 'I') Isotropic_scatter();
		  if(scat == '2') S2_scat();
		  if(scat == '3') S3_scat();
		  receive(bundle) ;
		}
	      el++ ;
	    }
	}
    }
}
// ------------ Add the walls -----------

void  MEDIA::addwall(WALL *w, const int num)
{
  wall[num] = w ;
}

// ------------- Iterate -------------

char MEDIA::iterate(void) 
{
  int i ;
  char ans, again = 'N';
  double percent, net, qsent, qrec ;

  if( (type == 'U')&&(iter_num < max_iter) )
    {
      iter_num++ ;
      for(i = 1; i < nel; i++)
	{
	  net = qm[i]* volume ;
	  qsent = (double)send[i] * bun_e ;
	  qrec = (double)rec[i] * bun_e ;
	  if( qrec != 0.0 )
	    percent = fabs( fabs((qsent - net)/qrec * 100.0) - 100.0 );
	  else
	    percent = 100 ;
	  if(percent > max_per)
	    {
              assert( (bun_e != 0.0) ) ;
	      ans = 'Y';
	      send[i] = send[i] + (unsigned long)(iter_fac*(rec[i] - send[i])) + (unsigned long)(qm[i] * volume/bun_e);
	    }
	  else
	    {
	      ans = 'N';
	    }
	  if(ans == 'Y') again = 'Y';
	}
    }
  return(again);
}
// ------------ Reset the # received ---------

void MEDIA::reset(void) 
{
  int i ;

  for(i = 0; i < nel; i++)
    {
      rec[i] = 0 ;
    }
}


//----------- Receive Bundle ------------

void MEDIA::receive(Bundleinfo bun)
{
  char absorbed = 'N' ;
  int i, hit, max = 0 ;
  double opt_l, opt_ext, l_ext, R, length ;
  bundle = bun ;
  assert( (bundle.optl < 0.1E40) ) ;
  assert( (bundle.optlext < 0.1E40) ) ;
  assert( ((dim [0] - bundle.pos[0]) > -0.1E-10) );
  assert( ((dim [1] - bundle.pos[1]) > -0.1E-10) );
  assert( ((dim [2] - bundle.pos[2]) > -0.1E-10) );
  if(type == 'C')
    {
      length = to_wall(&hit);
      move_bundle(length);
      wall[hit]->receive(bundle);
    }
  else
    {
      while( (max <= MAX_TIMES) && (absorbed == 'N') )
	{
	  max++ ;
	  if(max == MAX_TIMES) 
	    {
	      printf(" The bundle has been bouncing in the media for %i times.\n",max);
	      printf(" This may represent a problem with the model.\n");
	      printf(" Check Ke & single scattering albedo (Also see the manual)\n");
	      exit(0);
	    } 
	  length = to_wall(&hit);
	  //Optical length to the nearest wall
	  opt_l = length * Ke ;
	  // If the bundle is coming from another meida, its
	  // already defined optical length to exticition will be
	  // used. If the bundle is from a wall in the same box as the
	  // media, then an optical length to exticition will be
	  // calculated.
	  if(bundle.optl == 0.0 )
	    {
	      R = Random ;
	      if(R == 0 )
		opt_ext = 0.1E10 ;
	      else
		opt_ext = -log(R);
	      bundle.optlext = opt_ext ;
	    }
          assert( (bundle.optl < 0.1E40) ) ;
          assert( (bundle.optlext < 0.1E40) ) ;
	  if( (bundle.optl + opt_l) >= bundle.optlext)
	    {
	      //Interaction with the media
	      l_ext = (bundle.optlext - bundle.optl)/Ke ;
	      move_bundle(l_ext);
	      if( Random >= ssal)
		{
		  //Absorption into the media
		  i = find_elem() ; 
		  rec[i]++ ;
		  absorbed = 'Y' ;
		}
	      else
		{
		  //Scattering of the photon
		  bundle.optl = 0.0 ;
		  bundle.optlext = 0.0 ;
		  if(scat == 'I') Isotropic_scatter();
		  if(scat == '2') S2_scat();
		  if(scat == '3') S3_scat();
		}
	    }
	  else
	    {
	      //No interaction with the media
	      move_bundle(length) ;
	      wall[hit]->receive(bundle);
	      absorbed = 'Y' ;
	    }
	}
    }
}
//----------- Length to intersected Wall --------

double MEDIA::to_wall(int *hit) const
{
  int i, int_wall[3] ;
  double ln[3] ;

  //Find the vector length from the current position 
  // to the intersection of an orthoganal wall plane

  for(i = 0; i < 3; i++)
    {
      if(bundle.dir[i] == 0.0) 
	{
	  ln[i] = 0.9E50 ;
	}
      else
	{
	  if(bundle.dir[i] > 0.0)
	    {
	      ln[i] = fabs( (dim[i] - bundle.pos[i])/bundle.dir[i]) ;
	      int_wall[i] = 5 - i ;
	    }
	  else 
	    {
	      ln[i] = fabs(bundle.pos[i]/bundle.dir[i]) ;
	      int_wall[i] = 2 - i ;
	    }
	}
    }

  // Find the minimum of the three lengths
  int min = 0 ;

  for(i = 0; i < 2; i++)
    {
      if(ln[min] > ln[i + 1]) min = i + 1 ;
    }
  *hit = int_wall[min] ;
  return(ln[min]);
}

//-------- Move the Bundle by a given length -----

void MEDIA::move_bundle(double len)
{
  bundle.pos[0] = bundle.pos[0] + bundle.dir[0]*len ;
  bundle.pos[1] = bundle.pos[1] + bundle.dir[1]*len ;
  bundle.pos[2] = bundle.pos[2] + bundle.dir[2]*len ;
  assert( ((dim [0] - bundle.pos[0]) > -0.1E-10) );
  assert( ((dim [1] - bundle.pos[1]) > -0.1E-10) );
  assert( ((dim [2] - bundle.pos[2]) > -0.1E-10) );
}
//-------- Find Media Element at Bundle location ---

int MEDIA::find_elem(void) const
{
  int i, j, k ;

  i = 0 ;
  j = 0 ;
  k = 0 ;

  while(bundle.pos[0] > (delx * i) )
    {
      i++ ;
      assert( (i < 10000) );
    }
  while(bundle.pos[1] > (dely * j) )
    {
      j++ ;
      assert( (i < 10000) );
    }
  while(bundle.pos[2] > (delz * k) )
    {
      k++ ;
      assert( (i < 10000) );
    }
  return( (i + meshx*(j-1) + meshx*meshy*(k - 1)) );
}

//------ Isotropic Scattering ------------

void MEDIA::Isotropic_scatter(void)
{ 
  double theta, phi, value  ;
  
  value  = 1.0 - 2.0 * Random ;
  if(value > 1.0) 
    {
      assert(false);
    }
  theta = acos(value);
  phi = 2.0 * PI * Random ;
  bundle.dir[0] = sin(theta) * cos(phi) ;
  bundle.dir[1] = sin(theta) * sin(phi) ;
  bundle.dir[2] = cos(theta) ;
}

//----------- S2 Scattering ---------------

void MEDIA::S2_scat(void)
{

  fprintf(stdout,"\n\n The function 'S2_scat' has been called, but this\n");
  fprintf(stdout," version of GnuRad does not have a scattering model\n");
  fprintf(stdout," defined for S2 scattering. To include a scattering model\n");
  fprintf(stdout," edit the CLASS MEDIA, function 'S2_scat' in the source\n");
  fprintf(stdout," file 'media.C'. See the section 'Future Enhancements'\n");
  fprintf(stdout," in the user manual for more detail.\n");
  fprintf(stdout," The program has been terminated\n");
  exit(0);

  // Data which can be accessed by this function
  // (they are global variables in this class):
  //
  // Ke          Extinction coeficient
  // ssal        Single scattering albedo
  // p[0] thru p[5]  Six parameters, which are defined in the "Particle" 
  //             keyword. (data type double)
  // 
}

//----------- S3 Scattering ---------------

void MEDIA::S3_scat(void)
{

  fprintf(stdout,"\n\n The function 'S3_scat' has been called, but this\n");
  fprintf(stdout," version of GnuRad does not have a scattering model\n");
  fprintf(stdout," defined for S3 scattering. To include a scattering model\n");
  fprintf(stdout," edit the CLASS MEDIA, function 'S3_scat' in the source\n");
  fprintf(stdout," file 'media.C'. See the section 'Future Enhancements'\n");
  fprintf(stdout," in the user manual for more detail.\n");
  fprintf(stdout," The program has been terminated\n");
  exit(0);

  // Data which can be accessed by this function
  // (they are global variables in this class):
  //
  // Ke          Extinction coeficient
  // ssal        Single scattering albedo
  // p[0] thru p[5] Six parameters, which are defined in the "Particle" 
  //             keyword. (data type double)
  // 
}

//---------- Report the Media dimension -----
// 
// The dimension of the media is returned where:
//  i = 0 X dim
//  i = 1 Y dim
//  i = 2 Z dim

double MEDIA::dimension(const int i) const
{
  return( dim[i] ) ;
}
