// GnuRad 
// Copyright 1997, Ken Paoletti
// All rights reserved

#include <stdio.h>
#include "struct.h"
#include "wall.h"
#include "media.h"
#include "box.h"

// A box "has a" wall object (six of them)
// A box "has a" media object (even if its clear)

//--------- Construct BOX ---------

BOX::BOX(boxdata bdat)
{
  int i ;
  for(i = 0; i < 6; i++)
    {
      wall[i] = 0;
    }

  e_num = bdat.e_num ;
  length[0] = bdat.length[0] ;
  length[1] = bdat.length[1] ;
  length[2] = bdat.length[2] ;

}

// There is no copy construtor because the current design of GnuRad 
// does not require one. If there is a change it may be required and 
// will have to be added for GnuRad to work properly.

//------ Destroy the BOX ---------

BOX::~BOX(void)
{
  int i ;

  for(i=0; i<6; i++)
    {
      delete  wall[i] ;
    }
  delete media ;
}

//--------- Add Wall -------------

void BOX::add_w(WALL *Wptr, int wall_num)
{
  wall[wall_num] = Wptr ;
}

//----------- Add Media --------------

void BOX::add_m(MEDIA *Mptr)
{
  media = Mptr ;
}

//-------- Join walls and Media -------

void BOX::join(void) const 
{
  int wall_num ;

  for(wall_num = 0; wall_num < 6; wall_num++ )
    {
      wall[wall_num]->addmedia(media) ;
      media->addwall(wall[wall_num], wall_num) ;
    }
}

//------- Report its type of walls ----------

char BOX::wall_type(int wnum) const
{
  return( wall[wnum]->wtype() );
}

//-------- If the wall is virtual report which ----
//-------- box number should be joined. -----------

int BOX::joining_box(int wnum) const
{
  return( wall[wnum]->other_box() );
}

//-------- Return Media pointer ---------------

MEDIA *BOX::media_ptr(void) const
{
  return(media);
}

//-------- Join the two boxes, by exchanging ------- 
//-------- media pointers.                  --------

void BOX::join_boxes(int wnum, MEDIA *mptr) const
{
  wall[wnum]->addmedia(mptr);
}

//------- Report Total bundles received ---------

unsigned long BOX::received(void) const
{
  unsigned long r = 0 ;
  int i ;

  for(i = 0; i < 6; i++)
    {
      r = r + wall[i]->received() ;
    }
  return(r);
}
//------- Send All bundles ---------

void BOX::sendall(void) const
{
  int i ;

  for(i = 0; i < 6; i++)
    {
      fprintf(stdout," Wall %i sending all bundles\n", (i+1) ) ;
      wall[i]->sendall() ;
    }
  fprintf(stdout," Media sending all bundles\n");
  media->sendall() ;
}
//------- Report Total bundles sent ---------

unsigned long BOX::sent(void) const
{
  unsigned long sent = 0 ;
  int i ;

  for(i = 0; i < 6; i++)
    {
      sent = sent + wall[i]->sent() ;
    }
  return(sent);
}

//------- Print info  ---------

void BOX::print(FILE *fptr, char type) 
{
  int i ;

  if( (type == 'G')&&(e_num == 0) ) pelem(fptr);
  if( (type == 'P')&&(e_num == 0) ) plot_hdr(fptr);
  for(i = 0; i < 6; i++)
    {
      wall[i]->print(fptr, type) ;
    }
  media->print(fptr, type) ;
}

void BOX::pelem(FILE *fptr) 
{ 
  fprintf(fptr," ***** Element description *****\n") ;
  fprintf(fptr," Element Number - The element number as described in the User Manual\n") ;
  fprintf(fptr," [Resulting Temp - Printed only if it was previously unknown ]\n");
  fprintf(fptr," Net Energy/time - (Bundle Energy)*( #Received - #Sent)\n\n");
  
  fprintf(fptr," ----------------\n| Element Number |\n| Resulting Temp |\n| Net Flux       |\n ----------------\n\n");
}

void BOX::plot_hdr(FILE *fptr) 
{
  fprintf(fptr," # This file is designed to be used in GNUPLOT plotting program.\n") ;
  fprintf(fptr," # GNUPLOT is available on many platforms, please find more \n") ;
  fprintf(fptr," # information on GNUPLOT (via Internet) for more detail.\n") ;
}
//------------- Decide to Iterate ----------

char BOX::iterate(void) const
{
  int i ;
  char ans, again = 'N';

  for(i = 0; i < 6; i++)
    {
      ans = wall[i]->iterate();
      if(ans == 'Y') again = 'Y';
    }
  ans = media->iterate();
  if(ans == 'Y') again = 'Y';

  return(again);
}

//---------- Reset the # received before iteration ------

void BOX::reset(void) const
{
  int i ;

  for(i = 0; i < 6; i++)
    {
      wall[i]->reset();
    }
  media->reset();
}
